import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion, AnimatePresence } from 'framer-motion'
import { Music2, Calendar, CheckCircle2, ArrowRight, Piano, Guitar, Mic, Monitor, Star, Radio, Zap, Shield } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Rent.css'

const Rent = () => {
  const navigate = useNavigate()
  const [selectedCategory, setSelectedCategory] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const categories = [
    { 
      id: 'keyboards', 
      name: 'Клавишные', 
      icon: Piano,
      gradient: 'linear-gradient(135deg, #FF00FF 0%, #9D4EDD 100%)',
      description: 'Синтезаторы, пианино, MIDI-клавиатуры'
    },
    { 
      id: 'strings', 
      name: 'Струнные', 
      icon: Guitar,
      gradient: 'linear-gradient(135deg, #00FFFF 0%, #FF00FF 100%)',
      description: 'Гитары, бас-гитары, скрипки, виолончели'
    },
    { 
      id: 'microphones', 
      name: 'Микрофоны', 
      icon: Mic,
      gradient: 'linear-gradient(135deg, #9D4EDD 0%, #00FFFF 100%)',
      description: 'Вокальные, инструментальные, студийные'
    },
    { 
      id: 'monitors', 
      name: 'Мониторы', 
      icon: Monitor,
      gradient: 'linear-gradient(135deg, #FF00FF 0%, #00FFFF 100%)',
      description: 'Студийные мониторы, PA-системы, колонки'
    }
  ]

  const today = new Date()
  const maxDate = new Date(today.getTime() + 180 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (!selectedCategory || !selectedDate) return

    const saved = localStorage.getItem('musicRentals')
    const rentals = saved ? JSON.parse(saved) : []
    
    const categoryData = categories.find(c => c.id === selectedCategory)
    
    const newRental = {
      id: Date.now(),
      equipment: `Оборудование: ${categoryData.name}`,
      date: selectedDate,
      review: null,
      photo: '/images/rental-placeholder.jpg'
    }
    
    rentals.push(newRental)
    localStorage.setItem('musicRentals', JSON.stringify(rentals))
    
    setIsSubmitted(true)
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  return (
    <div className="rent-wave-page">
      <div className="wave-background">
        <div className="wave wave-1"></div>
        <div className="wave wave-2"></div>
        <div className="wave wave-3"></div>
      </div>
      
      <div className="container">
        <motion.section 
          className="rent-header-wave"
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="rent-badge-wave"
            initial={{ scale: 0 }}
            animate={{ scale: 1 }}
            transition={{ delay: 0.2, type: "spring" }}
          >
            <Star size={16} strokeWidth={2.5} />
            <span>НОВАЯ АРЕНДА</span>
          </motion.div>
          <h1 className="rent-title-wave">
            Арендовать <span className="title-gradient-wave">оборудование</span>
          </h1>
          <p className="rent-description-wave">
            Выберите категорию и дату возврата. Получите профессиональное оборудование уже сегодня!
          </p>
        </motion.section>

        <div className="rent-layout-wave">
          <motion.section 
            className="form-section-wave"
            initial={{ opacity: 0, x: -30 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: 0.3 }}
          >
            <form className="form-wave" onSubmit={handleSubmit}>
              <div className="form-group-wave">
                <label className="form-label-wave">
                  <Music2 size={24} strokeWidth={2} />
                  <span>Выберите категорию</span>
                </label>
                <div className="categories-flow-wave">
                  {categories.map((category, index) => {
                    const Icon = category.icon
                    return (
                      <motion.button
                        key={category.id}
                        type="button"
                        className={`category-flow-card ${selectedCategory === category.id ? 'selected' : ''}`}
                        onClick={() => setSelectedCategory(category.id)}
                        initial={{ opacity: 0, y: 30 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ delay: 0.4 + index * 0.1 }}
                        whileHover={{ scale: 1.02, y: -5 }}
                        whileTap={{ scale: 0.98 }}
                        style={{ 
                          background: selectedCategory === category.id 
                            ? category.gradient 
                            : 'rgba(255, 255, 255, 0.03)',
                          borderColor: selectedCategory === category.id 
                            ? 'transparent' 
                            : 'rgba(255, 0, 255, 0.2)'
                        }}
                      >
                        <div className="category-icon-flow">
                          <Icon size={40} strokeWidth={2} />
                        </div>
                        <div className="category-content-flow">
                          <h3 className="category-name-flow">{category.name}</h3>
                          <p className="category-desc-flow">{category.description}</p>
                        </div>
                        {selectedCategory === category.id && (
                          <motion.div 
                            className="check-mark-flow"
                            initial={{ scale: 0, rotate: -180 }}
                            animate={{ scale: 1, rotate: 0 }}
                            transition={{ type: "spring" }}
                          >
                            <CheckCircle2 size={28} strokeWidth={2.5} fill="currentColor" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <AnimatePresence>
                {selectedCategory && (
                  <motion.div
                    className="form-group-wave date-group-wave"
                    initial={{ opacity: 0, height: 0 }}
                    animate={{ opacity: 1, height: 'auto' }}
                    exit={{ opacity: 0, height: 0 }}
                    transition={{ duration: 0.3 }}
                  >
                    <label className="form-label-wave">
                      <Calendar size={24} strokeWidth={2} />
                      <span>Дата возврата</span>
                    </label>
                    <div className="date-input-wrapper-wave">
                      <Calendar size={20} strokeWidth={2} className="date-icon-wave" />
                      <input
                        type="date"
                        value={selectedDate}
                        onChange={(e) => setSelectedDate(e.target.value)}
                        min={minDateStr}
                        max={maxDateStr}
                        className="date-input-wave"
                        required
                      />
                    </div>
                  </motion.div>
                )}
              </AnimatePresence>

              <motion.button 
                type="submit" 
                className="submit-btn-wave"
                disabled={!selectedCategory || !selectedDate}
                whileHover={{ scale: selectedCategory && selectedDate ? 1.05 : 1 }}
                whileTap={{ scale: selectedCategory && selectedDate ? 0.95 : 1 }}
              >
                <span>Оформить аренду</span>
                <ArrowRight size={20} strokeWidth={2.5} />
              </motion.button>
            </form>

            {isSubmitted && (
              <motion.div 
                className="success-wave"
                initial={{ opacity: 0, scale: 0.8 }}
                animate={{ opacity: 1, scale: 1 }}
              >
                <CheckCircle2 size={40} strokeWidth={2} fill="currentColor" />
                <span>Аренда оформлена! Переходим к вашим арендам...</span>
              </motion.div>
            )}
          </motion.section>

          <aside className="info-panel-wave">
            <motion.div 
              className="info-card-wave"
              initial={{ opacity: 0, x: 30 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: 0.5 }}
            >
              <div className="info-header-wave">
                <Radio size={24} strokeWidth={2} />
                <h3 className="info-title-wave">Преимущества</h3>
              </div>
              <ul className="benefits-list-wave">
                <li>
                  <Zap size={18} strokeWidth={2} />
                  <span>Мгновенное подтверждение</span>
                </li>
                <li>
                  <Shield size={18} strokeWidth={2} />
                  <span>Страхование оборудования</span>
                </li>
                <li>
                  <Radio size={18} strokeWidth={2} />
                  <span>Круглосуточная поддержка</span>
                </li>
                <li>
                  <Star size={18} strokeWidth={2} />
                  <span>Премиум качество</span>
                </li>
              </ul>
            </motion.div>
          </aside>
        </div>

        <ImageSlider />

        <section className="features-wave-section">
          <motion.h2 
            className="section-title-wave"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
          >
            <Music2 size={32} strokeWidth={2} />
            Почему выбирают нас
          </motion.h2>
          <div className="features-grid-wave">
            {[
              { icon: Radio, title: 'Премиум оборудование', text: 'Только профессиональные инструменты от ведущих брендов' },
              { icon: Zap, title: 'Быстрая доставка', text: 'Доставка в день заказа. Работаем 24/7' },
              { icon: Shield, title: 'Техподдержка', text: 'Круглосуточная поддержка и консультации' }
            ].map((feature, index) => {
              const Icon = feature.icon
              return (
                <motion.div 
                  key={index}
                  className="feature-card-wave"
                  initial={{ opacity: 0, y: 30 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ delay: index * 0.1 }}
                  whileHover={{ y: -10, scale: 1.02 }}
                >
                  <div className="feature-icon-wave">
                    <Icon size={32} strokeWidth={2} />
                  </div>
                  <h3 className="feature-title-wave">{feature.title}</h3>
                  <p className="feature-text-wave">{feature.text}</p>
                </motion.div>
              )
            })}
          </div>
        </section>
      </div>
    </div>
  )
}

export default Rent
